#!/bin/bash

# Function to add a ban
add_ban() {
    local IP_ADDRESS="$1"
    shift
    local PORTS="$1"

    if [ -z "$IP_ADDRESS" ]; then
        echo "Error: No IP address specified."
        exit 1
    fi

    if [ -z "$PORTS" ]; then
        echo "Banning IP address $IP_ADDRESS from all ports."
        iptables -A INPUT -s "$IP_ADDRESS" -j DROP
    else
        IFS=',' read -ra PORT_ARRAY <<< "$PORTS"
        for PORT in "${PORT_ARRAY[@]}"; do
            echo "Banning IP address $IP_ADDRESS from port $PORT."
            iptables -A INPUT -s "$IP_ADDRESS" -p tcp --dport "$PORT" -j DROP
            iptables -A INPUT -s "$IP_ADDRESS" -p udp --dport "$PORT" -j DROP
        done
    fi
}

# Function to remove a ban
remove_ban() {
    local IP_ADDRESS="$1"
    shift
    local PORTS="$1"

    if [ "$IP_ADDRESS" == "all" ]; then
        echo "Removing all bans for all IPs and ports."
        iptables -F INPUT
        return
    fi

    if [ -z "$IP_ADDRESS" ]; then
        echo "Error: No IP address specified."
        exit 1
    fi

    if [ -z "$PORTS" ]; then
        echo "Removing ban from IP address $IP_ADDRESS for all ports."
        iptables -D INPUT -s "$IP_ADDRESS" -j DROP 2>/dev/null || true
    else
        IFS=',' read -ra PORT_ARRAY <<< "$PORTS"
        for PORT in "${PORT_ARRAY[@]}"; do
            echo "Removing ban from IP address $IP_ADDRESS for port $PORT."
            iptables -D INPUT -s "$IP_ADDRESS" -p tcp --dport "$PORT" -j DROP 2>/dev/null || true
            iptables -D INPUT -s "$IP_ADDRESS" -p udp --dport "$PORT" -j DROP 2>/dev/null || true
        done
    fi
}

# Function to list all bans
list_bans() {
    echo "Currently banned IP addresses and ports:"
    declare -A banned_ips
    local found_bans=false

    # Iterate over the iptables rules to gather banned IPs and their ports
    while read -r line; do
        PROTOCOL=$(echo "$line" | awk '{print $4}')
        IP=$(echo "$line" | awk '{print $5}')
        PORT=$(echo "$line" | grep -oP 'dpt:\K\d+')

        # Initialize or accumulate ports for each IP
        if [[ -n $IP && -n $PORT ]]; then
            found_bans=true
            if [[ -z "${banned_ips[$IP]}" ]]; then
                banned_ips[$IP]=$PORT
            else
                if [[ ! "${banned_ips[$IP]}" =~ $PORT ]]; then
                    banned_ips[$IP]="${banned_ips[$IP]},$PORT"
                fi
            fi
        elif [[ -n $IP ]]; then
            found_bans=true
            banned_ips[$IP]="all ports"
        fi
    done < <(iptables -L INPUT -n --line-number | grep "DROP")

    # Output the collected banned ips and ports, or notify if list is empty
    if [[ $found_bans == false ]]; then
        echo "List is empty."
    else
        for IP in "${!banned_ips[@]}"; do
            echo "$IP: ${banned_ips[$IP]}"
        done
    fi
}

# Ensure that the user has sudo privileges
if [ "$(id -u)" -ne 0 ]; then
    echo "This script must be run as root or with sudo."
    exit 1
fi

# Ensure correct usage
if [ "$#" -lt 1 ]; then
    echo "Usage: sudo $0 add/remove <ip> [port1,port2,...] | list"
    exit 1
fi

ACTION="$1"
IP_ADDRESS="$2"
PORTS="$3"

case "$ACTION" in
    add)
        add_ban "$IP_ADDRESS" "$PORTS"
        ;;
    remove)
        remove_ban "$IP_ADDRESS" "$PORTS"
        ;;
    list)
        list_bans
        ;;
    *)
        echo "Invalid action specified. Use 'add', 'remove', or 'list'."
        exit 1
        ;;
esac